#!/usr/bin/env python3
import argparse, csv, glob, os, re
import numpy as np

def find_L_dirs(root):
    for d in sorted(glob.glob(os.path.join(root, "L*"))):
        m = re.match(r".*[\\/]L(\d+)$", d)
        if m:
            yield int(m.group(1)), d

def load_counts(ldir, L):
    fname = os.path.join(ldir, f"flip_counts_L{L}.npy")
    if not os.path.exists(fname):
        raise FileNotFoundError(f"missing {fname}")
    return np.load(fname)

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--root", default="data/results/vol4_loop_fluctuation_sim",
                    help="Root results dir containing L*/flip_counts_L{L}.npy")
    ap.add_argument("--out", default=None,
                    help="Output CSV path (default: <root>/flip_counts_summary.csv)")
    ap.add_argument("--hist", action="store_true",
                    help="Also write per-L histogram CSVs (50 bins)")
    args = ap.parse_args()

    root = args.root
    out_csv = args.out or os.path.join(root, "flip_counts_summary.csv")
    os.makedirs(os.path.dirname(out_csv) or ".", exist_ok=True)

    rows = []
    for L, ldir in find_L_dirs(root):
        arr = load_counts(ldir, L).astype(np.float64)
        n_links = arr.size
        expected = 2 * L * L
        ok_size = (n_links == expected)

        # stats
        total = float(arr.sum())
        mean  = float(arr.mean())
        std   = float(arr.std(ddof=0))
        mn    = float(arr.min())
        mx    = float(arr.max())
        p50, p90, p99 = (float(x) for x in np.percentile(arr, [50, 90, 99]))

        rows.append({
            "L": L,
            "n_links": n_links,
            "expected_2L2": expected,
            "size_ok": ok_size,
            "sum": total,
            "mean": mean,
            "std": std,
            "min": mn,
            "max": mx,
            "p50": p50,
            "p90": p90,
            "p99": p99,
        })

        # optional histogram per L
        if args.hist:
            hist_counts, edges = np.histogram(arr, bins=50)
            hist_path = os.path.join(ldir, f"flip_counts_hist_L{L}.csv")
            with open(hist_path, "w", newline="", encoding="utf-8") as f:
                w = csv.writer(f)
                w.writerow(["bin_left", "bin_right", "count"])
                for i in range(len(hist_counts)):
                    w.writerow([edges[i], edges[i+1], int(hist_counts[i])])

    # write summary CSV
    fieldnames = ["L","n_links","expected_2L2","size_ok","sum","mean","std","min","max","p50","p90","p99"]
    with open(out_csv, "w", newline="", encoding="utf-8") as f:
        w = csv.DictWriter(f, fieldnames=fieldnames)
        w.writeheader()
        for r in sorted(rows, key=lambda x: x["L"]):
            w.writerow(r)

    print(f"Wrote summary: {out_csv}")
    if args.hist:
        print("Wrote per-L histograms alongside each L* directory.")

if __name__ == "__main__":
    main()
